<!DOCTYPE html>
<!--
/**
 *
 *       dilbo - digital logbook for Rowing and Canoeing
 *       -----------------------------------------------
 *       https://www.dilbo.org
 *
 * Copyright  2023-2024  Martin Glade
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
-->

<html lang="de">
<head>
<title>dilbo installer</title>
<meta charset="UTF-8">
<style>
html, body, input, textarea {
	margin: 0;
	font: 1em "Prompt", sans-serif;
	color: #1c1d1f;
	background-color: #f0f3f4;
}

h1, h2, h3 {
	color: #033161; /* headline color */
}

input, p, textarea, blockquote, code, h4, h5, h6 {
	color: #1c1d1f; /* text color */
}

a {
	color: #033161; /*background may change */
}

a:hover {
	text-decoration: none;
}

a:visited, a:hover {
	color: #0063a0; /*background may change */
}

.formbutton {
	color: #f6efe7 !important;
	background-color: #033161 !important;
	text-decoration: none !important;
	cursor: pointer;
	border: 1px solid transparent;
	border-radius: 0;
	text-transform: uppercase;
	font-size: 0.9rem;
	padding: 8px 6px 8px 6px;
	line-height: 120%;
	box-shadow: 0 3px 2px 0 rgba(0, 0, 0, 0.2);
}

.formbutton:hover, .formbutton:focus {
	color: #f6efe7 !important;
	background-color: #3a3b3f !important
}
</style>
</head>

<body>
	<h3 style='text-align: center'>
		<br> <br> <img
			src='data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAEAAAABACAYAAACqaXHeAAATSXpUWHRSYXcgcHJvZmlsZSB0eXBlIGV4aWYAAHjaxZprctw6soT/YxWzBOINLAfPiNnBXf58CbJbki2f4+OZiCtZYovNBoGqrMws0Gb937+3+RdfMYZgQswl1ZQuvkIN1TVelOv+6ue3vcL5fb7C8xZ/fzlv3m84TnmO/v6zuuf84jyv7fN3fW5iX9e/Bnq9sI1X8eON1p7z/ev5/gzoyo8DPTPw9r7zNZ8PPAN598zomfp4ZpRqyV+WNsdz5/CcKh8/wWeXYrI58Du4K+dUeV3cFTLxnJroHq6egeId0PeJ19+vSx1zcstbf/Hb+2eWXj/Rt3Ns/BSjC88f0Vd+ex9P4C9SyRSYaX1u1K53MD/H5iNGv/j6nWVd3GSvGxPmx5F/xM37lf3F+QcG76yV9Lzhv6b1Su/jt+dtfA30esO/7+M+37mM952/nLfJhs+hMJ/Tvfcs+yyaVbSQiEV6FvVaynnFdV1RPJ9KfOcrGVBbeKHvyne52jW417wGldZ5Xa0j99sGO22z265zHHYwxeCWyxydG8b5c7KQpOoGuLA+6NtulwHE9AVwjIOh4N17Lvbctp7bDVuuaa5pudRZBrMHUX/4bX73wr1VS9Ze5R0r5uVUnczisqRfBy4jB3Y/QY0nwK/vH7+UV08G4wlzYYHt6uYeokf7AS5/Eu25MHK8k2zzfAYgRNw6MhnrycCVrI9gwVzZuWwtgSwkqDF154PrZMDG6CaTdMH7RHKoDu7NZ7I9l7ro7tOwqg+GYk0+kxtKlmSFEMFPDgUMtehjiDGmmGOJNbbkkyovpZxEzy37HHLMKedcTK65FV9CiSWVXEqppVVXPfQdK3VaS621NW7aGLnx6cYFrXXXfQ899tRzL72a3gbwGWHEkUYeZdTRppt+UuAzzTzLrLMtu4DSCiuutPIqq662gdr2O+y4085ml113e2ftSetP3/8ga/bJmjuZ0oX5nTXO5vwawopOonJGxlywJDyTNTIGsJWzq9gQnDKnnKFHokvHJKOSM60yRgbDsi5u+87dkznj2v8mbyaXkzf332bOKHW/mbmf8/Zd1qZUYpyM3WWooF6e6uP9VZorTfL6y6P5uwt+9/j/MFBv0GWaew270h5zJX7nNJK3dW9r2kw1uhpDbrbO0KotiWvIVYt7yyzk2ndfBHb0kfe1YOCxORVDbSOhFrOSBLOXLh9ze2p4X2nqFmvlmXosPeUeQdjyHSRlZbEzzl49tjWyy9i0lnexfRkbM/4hV1KoDJWqbOc14mJCo7dde995nduBBAsS1uYuY7heUug1heV2DibHEUEfdkjhwJGMP4u4+c1IdxaQ+7523sv6sdd0mmOp7VmAWVoas79XcJZHfF8rqC1iQ6zPaYaRKQi/uztBIGt7zrJmGcm1GU1qihzDJ7+r3avalckTwxafMFhlZ88VNcY1kGAKelqE3bWkcsk1LU7yz0gOWk+OgiuTDKS6a6aEKtFeuac1VrYUlKo1E1WV6wUpCCGBsnM7UNzNGQWAjM9wBxsr8WdH8/0bPcI4TK9PxaOPWXu2swPFyeIcqxpQC4wV44SoZt1mNiVi+wQUATy4hRXCbHFCCORqMASx6ucNopqDX7ZjTgQ/P3dqulXwZvcAGsnGElgnefZg1gO5iQHt2Y085xzEQgXmZk5+YT4iRhIM+O26JhK3oZSwVpoJ+dH4aIl1uUQBxi6yrTs2ct2ADyvlLzc6C5hdGa2klKuaCatbLqokfS0/xHoHJ9SbRl/DTxY0mWtmKGa1tVTSmTrkPDFih/uTWXk0YgZ2lqsbSlRNBVX0Xo5lpANVJl9OPZ+bEA2WXjyxKGOGCtE7M9q1HK5/2zJWqjUWm7VcYkToutJFSJWuEFNrrY4cJ8Xsmc0iaSfXfRQDW/2XCLqP5suJVZzUY6n4oCRyCHhsaeGvqgwGLLkZO6jOwPyL6xNu3OS3X6oeG2fI3iJg/iQEHrwTIhJ8EqKwCS4zmQ4q9jV3KZl85DUj4tMX4PI3uNJH9vqwe4BNKxrZIHBv2IEyp5KNRq4wqpJKagvVufOsifHQzAwX7DoBCsxC6klWTX27uRVr5goOWhq+kX5gXAbj+l16HlckV8ebDvj9gZ8nJgcWrUM4V8tlETGmOTP64RCGPIwKTnegRj0hnky0Bmrp6stZ1dOVOdn1ga61A0yCBNXASaCScmbqiITxfLi3mPxgqr3XNivkQ1xrR36UKRbE2qYoU3nqN/de5Av5gj9TR8ScwVuMctPzKDP8sdaaH090mJyor6HlSBsFB6SO+3TZmK0avdFE/YGmeqinG4zLLPDoTjBTgALw3yTb7VoeMJG6EuWr/lJ7zRFf1JR0YNsgOHQWLAULk/jhiXThJsyODPg9lnKjTG4pA7U4NqKPpzMwepsXwjN8hWDjBnfhWAUG3rkfHBW4iXWMlM8HMXVgBYh0DYqr5W4HkNN6hSWkfcPzRGbNDiQt42aA6RfZlyp24NBEELBivWc3z9LAehIxwmml0Sn2MDCHqtAY/xoAFCcSES3ngYwBAv8TBHyn/Qdqdf+sL+EX52Pf0RC5TDGxFpioLBbJ2rP45tHi3Bym/37VdtEmSuw34VKDHroCB3uZiEAjg6Q8QLWYmxWprxVURAhKCQoXELk/JS7ylPOuwFRWQ5+0OzaIbUHJFu4XdTFyoXShfFmUCNqySraRf0SxKAdWueSyeLzGcyHQMAN/meQ3UDHaBHUSvF1tx2uM0tSC0JKf9OFHlD4qg/RRGal8Om8+v9HAGoDBqoMqiAr1aTSphGy97Ut2QOt+qd2r99H8eOK3j0T8NhrhkJ5R/B6xGBFqYb6tSPiYXoxqRmiCDjB2HNs/lhuaH7tQgvRclsXlbahoggwQX8WIUIDVTUqgyhamo4vyQZ4k0PEUWPW4Ae5M3R4nMg7VtoMCuimJRc43+0c6OydzNIvf6UgPhXiK/7hMtO+2hSdNafFlgMgF5aJuKvP6WS+wBZNikn3J24EjxIj7Qtz7thj2Fik6h4wZjdm3SD8wbdDIgiZACscpAT8g0YYdUhlRhsOu4gHUqYlEPUATVeRu1mHRC9mlJSSOwM/OS55VPHSTJEow299k0vyzlEu/hWX0jKqjNKarAG0FTATWuW5+D/TxAxhfUEEpnfQzewajc8Zc0fFSTMd/1DUXkk3RdeKqmMi3KjgXKZ805D6U6aQbeax98nXcN4V6zaMe8AjiPyFVazD8hILig2HG3tCCfAKZI3GYjIakIp7+QQIWFnR8JwLmkHmtkvMRvSzxTqw+yK+SbH/bXSU7ARSGWaH6HLVdmsFru1j6plczpeYH5w5Wd/mG+Qt2B+bYUGbFGljxcYXhdoX8iILuINIdFfyhJ5RUdWkRMrGF7sQBgKfeaaXyDYRwU+V3R/N3F3w+3j6Nnm3Zu4FV5zlv/TXWB619FkVEeaEAaG+fLKFkiiUraNMT5uZO7xbrqD8IgfmsBFRzG08/y7uMRh4vYR/SLbA6AUggotCFNJp2JdVVqPZ62qzk3cXQAb+eC90KEFCZLzzfIebTJruj2XeTXCT4XAsHO7AiaTDSBhrwwzHrQ1LgrEL1R3X19gaEzMBRkQOIpw0g1/KVZA1TObcbYVWUh4Zz1jKBFMICVKnuPRH7pe6eJexHVbZLMm3ukMDMIgHDb/Uo8jgYkwrVhjxGZRWgIwKBXzQUd/NAmCEjhwE0Ocu/u7f6V1f+SPzNd29g4hMaax+HBuQxJHAnpYvYvswShXVvfRDwgK75E3QChXOpLFRFe3zhIUsMQSN3U/CCPuXqzxp6a4A9O6oFX4jMGE/xQRnLnUZiayPs1DWeatSboAeAFGv4mzWKXV3OAubPHzsF5qetAqh4Wb+t9gCYphoSBz5h3rs56Y/ekHkxf8NRyV8amQk1uFNWTv2t/ehvAZ3PmvwgG6AQYoQDD2yRRPjuhm3XjoTJL9xq8gTBSiXwlLaccc9GisBxtE0fxOjlVpK4+1BiVOzGNlsTHvFwYhYn0hDIL1NUFYGquJI1AJQr2g0VxtXuTptVhpfqg2nAz6a8fQ0gDdoUQSKXHA/qDe74h5aSwIAwa8IiN8X1tEForZZQcja/oLwASity0QIuNyyaSAhQG623gdZKj1bWeAtfxPpB+nUUGCTj+DCEjypWpOEfSJ75nQsx09eqKtezXaXuuKpNOxRA87J66AZ8wICl4QEQtUavbDuBgASRtba0Gdx2om9Uq3c4M6pgTjbPvsYNrmVIbO6HGixoQQhsQPohVIszwA9RSskeXD77MnoWNNrdd0qbITOCnDFaN0bIragLl6OOyMnbeG3DcJksJiFbV5T9lV+a/fDUGG+zfNLvhXdqlM4NndzAiZELnVA55VG1lP7RwoAbeOnZJdGTKsx2i4YC1vZRjWM1PVvUrM72GrhL2sjkR4iij4vlpBNk0fVgNPHDDZMlo1kfXYvVtvJPFO7rMU74iN9zR8DFra9MWBOSDbuXtqhTgCi3ynovkpxXyE67NpyJR9SO91B3s80talU7ivmB+Lghfhtm/MElkE58RWoZj+S7Y3Q8ljZKQ6VqZ8u0ELIZWTolHwwReu08URHaPsHxLz0nRLLgQVhkhFYCvUNDJq4hGqCKXCTERle3q8/eUG7CWm+BPdEFgRZVOHsbZAgeJL1NKRNh1ONrMWsYeWyNtbbg9rG2rgINp6ZPXHqwhV1zB7gY07PfoeFp6BBIv85OFmpMrkGJ8Z1OFksjEYIq1kjPLt/Z3V5nwLgavKfwltsz5NPyPLv0erw9IX98NP1bbd3W0PVcqK2CXYaNWT04xPypiC7vCCSFieCefFtEA66GfMqljahPyT6dIg3eqicmS1WkeBJwFzpiFnpJjgJebl6jNzW/BwaCrnkmus9EY5QhQK9QfkrKYlree+/yi9/t0k099k+4EcqBqqXV26uTAPwOQsQl9BfOEybCYimcZaO2piVzQUZhnlm7ha2i+lc3iSXvAnqwPtpsoCmiIJMK/7iF5Q+CG+9q/XqanEUeRCMADSie3Mv6HXU/O2NQnZcFGtoHc+MRzX1cVbaVpCNELuLIChfizpZWUY92xk6bBa3fTRNSISONLEYmUcpYdBeelFdheZUsT8GK0t2QI+uwkXyF1mseS+fvbRgv29BkxBcYgiU74qodpn3Y8FToOER7do3OpkyMeoBnqp69xTTgFUvDWSW5keWRuxTQSG0qeogLk5My1QJ3qapv0mVGY8hlpGFe6IQQl2roDaQDI6afW+u/oIYMNeRIeVNvCCSIbE39IPxAQuTpGhoezwY2Zmjfe/Upyam8t98QqS8bcMbVeOlZlrZJo8M+4S89I+kT2MyzYefPQHKZalO/3/41wrkVbWH/bQHw2m/TDizmAVCra6SssemCk2q4IeRURvqha8PVXsLjsvveW896ygIs8723nt976/7Zl+kVR0szj/5Rv0Q7B5x+vgE5bif0Rld+g8s/4NqnmVzHx6mvF3xpWny+dwJqNh+sgpem2oCFHg5guunPRhOmuGW6Gr3lomsbe9A29qQHTQxWxZc3INvZj9RmNklkPZCOSg8YUEln95GU37vHT+P5mE3eI0mOFiglX41A3aDQhvKTQV86PFm1Y6fnURZvK+We9jwgGVgzJNrt49P6awpROKJFbWfnUfvJ2Mutqt3HlB76hx4p+oKfcvT3qCpCIvmCdm98322rNTcP4WBtwDEGImndxeeqnhGlo4c5fnwy3M9q7Kk3PpivgePBP5sLjknBW4/TtKmkyIhR4v7aKP68TSz+5EKnLaANe5bWO0BEOFFauRhZ3XHbG5rE/N0zjUiJoAcor3ptPWbJ2Dis8d3tAsgGIMX66YRU04Zg8VIvMh35cVPqHNVCRNwpYGCCzSF5dgeZCfNsxpSXX3wSigCmk1A0MI8QaVTU/NDwzvtm6WxnP7Bg3eahxtz1xIWxlset5RxcqnrQSA4jFVzqeaxAv6tS69xoqdQ+KDHRiqJu6FY+sLoncaEK905aUk++6nnqd4xgOn5jEeROc8X6KTlIeEzTiVTXMwp/U+jQDomme5o+YAit2Vyc196l3g/D3xFrsu3QadRMqDWbuzhT5eXQ34EC+zMUkCCX5SSA61OhmIvcwZafkCitQ7OkA1IwqAxu/1I7ku5GpSVRLexI4an/VEl8bSt5v9HVeHXZJ9UwwW20NGW4h8TIJ2mXRQiAZqce7R+3ObD3LSTMJp26dj57YN0Att79KTQSY7031/vP2q6urOk5jyY25WPr196QElePUaZBOvZFpMuxVKQejRCkabSsiA7vXUUjTQ2yFq/nXrnTHiMZEhgvgWGgLoU5e47dgxjmXBPRovWs6o3Bq5CLk1c4TlcLy77Q8BixlFrDROBN/TkPBg9tHgQpsfwd8bOl3KkouBKG9Xili+bMvwpKLdIytdzP+GO8XCbzz5Cs51b3C5oZDSMxnP5/xlADcp5WZgjgOIEuR8JAVc3lzff1eUy41IdP/GrWdh9+XK4jKjhyphWnA7rxCFha1Osch0EUHQPk6+x3ldNy+Z3q6z8qnMJ6lCLfj30OxmQtmnZ/tqdTdd3YNDFqFU1Mr/9ecf1JV2L+rI35crQqOfMfJn3npfYrtN0AAAGEaUNDUElDQyBwcm9maWxlAAB4nH2RPUjDQBzFX1OlIhUHK4gUyVCdLIiKiJNWoQgVSq3QqoPJpV/QpCFJcXEUXAsOfixWHVycdXVwFQTBDxB3wUnRRUr8X1JoEePBcT/e3XvcvQOEepmpZscYoGqWkYrHxEx2VQy8QkAY/ZjBkMRMfS6ZTMBzfN3Dx9e7KM/yPvfn6FFyJgN8IvEs0w2LeIN4atPSOe8Th1hRUojPiUcNuiDxI9dll984FxwWeGbISKfmiUPEYqGN5TZmRUMlniSOKKpG+ULGZYXzFme1XGXNe/IXBnPayjLXaYYRxyKWkIQIGVWUUIaFKK0aKSZStB/z8A86/iS5ZHKVwMixgApUSI4f/A9+d2vmJ8bdpGAM6Hyx7Y9hILALNGq2/X1s240TwP8MXGktf6UOTH+SXmtpkSOgdxu4uG5p8h5wuQMMPOmSITmSn6aQzwPvZ/RNWaDvFuhec3tr7uP0AUhTV4kb4OAQGClQ9rrHu7vae/v3TLO/H9Ihcs19gLPWAAAN/WlUWHRYTUw6Y29tLmFkb2JlLnhtcAAAAAAAPD94cGFja2V0IGJlZ2luPSLvu78iIGlkPSJXNU0wTXBDZWhpSHpyZVN6TlRjemtjOWQiPz4KPHg6eG1wbWV0YSB4bWxuczp4PSJhZG9iZTpuczptZXRhLyIgeDp4bXB0az0iWE1QIENvcmUgNC40LjAtRXhpdjIiPgogPHJkZjpSREYgeG1sbnM6cmRmPSJodHRwOi8vd3d3LnczLm9yZy8xOTk5LzAyLzIyLXJkZi1zeW50YXgtbnMjIj4KICA8cmRmOkRlc2NyaXB0aW9uIHJkZjphYm91dD0iIgogICAgeG1sbnM6eG1wTU09Imh0dHA6Ly9ucy5hZG9iZS5jb20veGFwLzEuMC9tbS8iCiAgICB4bWxuczpzdEV2dD0iaHR0cDovL25zLmFkb2JlLmNvbS94YXAvMS4wL3NUeXBlL1Jlc291cmNlRXZlbnQjIgogICAgeG1sbnM6ZGM9Imh0dHA6Ly9wdXJsLm9yZy9kYy9lbGVtZW50cy8xLjEvIgogICAgeG1sbnM6R0lNUD0iaHR0cDovL3d3dy5naW1wLm9yZy94bXAvIgogICAgeG1sbnM6dGlmZj0iaHR0cDovL25zLmFkb2JlLmNvbS90aWZmLzEuMC8iCiAgICB4bWxuczp4bXA9Imh0dHA6Ly9ucy5hZG9iZS5jb20veGFwLzEuMC8iCiAgIHhtcE1NOkRvY3VtZW50SUQ9ImdpbXA6ZG9jaWQ6Z2ltcDo3YTAxNGU0Yy05N2E4LTRhOWUtOWViOC0wOGRiNmMzOWFiOTIiCiAgIHhtcE1NOkluc3RhbmNlSUQ9InhtcC5paWQ6MGVmN2JhZDktM2Y2YS00Y2Q3LThlYWItNTVmMjVhNjliZWEwIgogICB4bXBNTTpPcmlnaW5hbERvY3VtZW50SUQ9InhtcC5kaWQ6ZDgyZWViZWQtMTYzMy00Yzk4LTg1MTAtNjEyNjgwYWFhNzZlIgogICBkYzpGb3JtYXQ9ImltYWdlL3BuZyIKICAgR0lNUDpBUEk9IjIuMCIKICAgR0lNUDpQbGF0Zm9ybT0iTGludXgiCiAgIEdJTVA6VGltZVN0YW1wPSIxNzQwNzc5OTc2NTI0MzMzIgogICBHSU1QOlZlcnNpb249IjIuMTAuMzAiCiAgIHRpZmY6T3JpZW50YXRpb249IjEiCiAgIHhtcDpDcmVhdG9yVG9vbD0iR0lNUCAyLjEwIj4KICAgPHhtcE1NOkhpc3Rvcnk+CiAgICA8cmRmOlNlcT4KICAgICA8cmRmOmxpCiAgICAgIHN0RXZ0OmFjdGlvbj0ic2F2ZWQiCiAgICAgIHN0RXZ0OmNoYW5nZWQ9Ii8iCiAgICAgIHN0RXZ0Omluc3RhbmNlSUQ9InhtcC5paWQ6YzBlYTJmYWUtM2QxYS00NmU2LTg4ZjMtZDg3OTAyMDc5Y2VmIgogICAgICBzdEV2dDpzb2Z0d2FyZUFnZW50PSJHaW1wIDIuMTAgKExpbnV4KSIKICAgICAgc3RFdnQ6d2hlbj0iMjAyNS0wMi0yOFQyMjoxODo1NyswMTowMCIvPgogICAgIDxyZGY6bGkKICAgICAgc3RFdnQ6YWN0aW9uPSJzYXZlZCIKICAgICAgc3RFdnQ6Y2hhbmdlZD0iLyIKICAgICAgc3RFdnQ6aW5zdGFuY2VJRD0ieG1wLmlpZDo0MTdkMjBkYi1lZjg1LTQwZjAtODZiYi00ZjM0ZTM1YzEzNGEiCiAgICAgIHN0RXZ0OnNvZnR3YXJlQWdlbnQ9IkdpbXAgMi4xMCAoTGludXgpIgogICAgICBzdEV2dDp3aGVuPSIyMDI1LTAyLTI4VDIyOjU5OjM2KzAxOjAwIi8+CiAgICA8L3JkZjpTZXE+CiAgIDwveG1wTU06SGlzdG9yeT4KICA8L3JkZjpEZXNjcmlwdGlvbj4KIDwvcmRmOlJERj4KPC94OnhtcG1ldGE+CiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgCiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAKICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIAogICAgICAgICAgICAgICAgICAgICAgICAgICAKPD94cGFja2V0IGVuZD0idyI/Pj8YT3EAAAAGYktHRAADADEAYaH/H8YAAAAJcEhZcwAALiMAAC4jAXilP3YAAAAHdElNRQfpAhwVOySHzDHfAAAAGXRFWHRDb21tZW50AENyZWF0ZWQgd2l0aCBHSU1QV4EOFwAACZ1JREFUeNrtmnlclVUax7/nvsAFFC4CLqCyuoJLV03R1BqrSSeXT2rlkstVxxbHT6Zt5jiVOqXZOJWT5ZKvazNjo4ZLZpk6FS6IvWSaSYCBC7gAAoIs971n/hAdI15A2ZXff/ee57zv/f2e55zzPM+5UI961KMe9bhzIeryj297zxhzSp4yyy7FGCdBrtnE9AuH1J13hACK1eahwB4dul77zgS6pxMz0g6p797WAihWmwJsBgaVNO6hsDTQQ0z9fs/KwrKeZaqjAbDAiDxAts6TiVlyR3DE2IZlilkHvT8RmF+WXaEkxC6Fnp8Su+e2iYBDJy/1mfpIzyXltZeITrdNBMT8khkihNjZoWOYJbiRC7sPx5U5x1XID/JSYg/W+QhQrDav+Lj4rUBTk8nEg/3vZ+lfbPianUqbluqmsKbOR4BitTkDn365NyYiPMgX/+b+ADTza0afbu04EXuUczkFxaflAQOyDq+Kr/MCmPysi4ERBQ7J9m+P0KKhIDQ0BGESWLws9OllJS0pkfhzWf9f+jBe19Qd5RK4lnv/T8CrN363V4vHOTedsPC2ODk54ebmRo8eVkzZaWgJqQDzdE19r9zvqMXkHwLWlLRPxcSdJedsMhE9uyKEwNnZGWuXjvi7i01fxyZMkamxsk5ngorV1h7YB3iVfLyBOncyYeHtSU9Lx8fXByAG6NstyHLlppZYLSTvC2w1Ig/w1tShhHcII+qbffLJ5xZw/NjxZGDwzZKvdRGgWG0uwJdAXyOb54b35vFRw4j7KU6OfeUDAeQAfXVN/e6WNtlaFgAflkZ+RO92DHtsCOdSz8nZ81cKwAGMvVXytUoAxWp7HrAZjfdt3YRJk0eTl5cnF7y9QiRl5QP8WdfUTRU6ZmsJ+cGlFThBnq48P30iZrOZpR+sZV/iRYC1uqa+WeE8oxaQ7wysNzqSnYRgzswJNG7SmI0bIuUn++MEEAX8sVISrRom3wzYAhjW7W899yht27Xhv3u+4Z2NUQI4CQzVNTW/TgugWG2uRV2dACObF0feR6/ePTl65Jh8+R+bAbKAwbqmnq+0VLuGyAtgJRBhZPPEfeEMGTqQs2fOytkLVgnADozSNfVopdYaNRQAs4GRRoP92vsxYdIoci7nyPkLV3A2txDgRV1Tt1d6sVUD3n+seIFzI9p4uzN92gScnZzkkiWrRXRyhgCW6Zr69yqpNquZ/N3AKqP3mk2C116eiI+vD/9cv5FPDyUC7AamVlm5XY3kWwKfAm5GNm+/MIrQ1qHs2rmbJVujBfAz8KiuqQV1WgDFamsARAL+Rjazxj1I94i7if3ue2Yv2waQAQzSNTW9Shsu1UDeBKwFrEY2tgc78/Cg/iQnJcuZC9YAFBZ5/kSVd5yqIQD+CjxiNPhQxxaMs40gKzNLzpu/QqQX6ABTdU39qjqi01TF3h8LvGQ03qGpB9OenQDA4sUq36dmAbyna+rS6tqbTFVIvjewzKjn0MhFYfZLk7B4WVi/5hO5PTZZADuAGdV5MpmqiHwwsBEwG9m88cITBAUH8vn2LxzLP/9OAMeAEbqm2uu0AIrV5lnU0mpiZPPapD/QpZuVmOgYOUfdaQIuFOX4WdWdmFWqADG/ZIp+7fzWA+FGNk893I2HBjxAYkIiM9/+WAD5RdVdYk3k5JUdAdNmzZoycGSf9iUODuoSxMgnhpOeni5ff/MjsgodAE/qmvptjVWllfWgoRNfbdP5rvCNHp4eTl27dcZL5LH/WPL1cau/hVdmPo3JZJILFy4T0UlpAPN1Tf1bjfYkKutBp03BL/4Ue/Teuzq1wWKxEN6hPW2aNuDLg8dp6ubMgtefwdvHm4+Wr2PTwXhR1At4+mYuMaoCldIWD+891iUhVzldKGkcbHFl7syJtG7bGoAfj/6Iw+EgvGM4kZu28ea6rwA0oI+uqTnUMColAuyNrcOuSDEO4FK+ne27YwgL8KZFy+YUFBTQzK8ZR2J/kK8siRRACnC/rqkXqQWoFAEUv7sW2RGh1z7rEnZE/YCXuMLy1Vto16o5Ia1CRE7KqSvHz2T01zX1OLUEFV4C/nePbXXerpyQBifKva2bMHfuDMxmc4Gu68N7hHpvpRahwhFg8rPOKJDC8DYnKT2H7NNJGV27dRrXq03jSGoZKhQBnfqOdo677JJcKGlm8HCHm0mutCiOmaej11ykFsKpQkdfrvMgI/IugqMNTY6pF2JW782m9sJJsdpaF6Wu9wCtuXotfR44ztV+3Le6puolTc53iKdKWFNX3E1yjkWxL0qKXldmK8tsHe9qR/gBPkA/oAPQvChFTgSigR26pl6okiWgWG1lJSKngfeB93VNve7Mlt3HBqYWKgmOG/YRNxM7LIpjypno1SdLVd1qawuMl/AwEFaOvaigqKU2vyI3wTfCrcv4BgVSDBeK1SYfCPMnvH0wUQePEXP6ktGci8Ac4ENdUwstXcfPu+wQswCcBCkNTY5paTGrN5RBvBMwR179m+uvTo2WDV0Y3M/K+YsZbNgXZ7Q5yaIy+6VbLZ5crDZ3BzwrYTrgKzbuPSIDAgMQQqDrOgnxiXLb1l38O+qE0QZ5tKEipuU7pGqXNHc3yaW+zvrLiQfWZpVSIvsCbwATinu7e4A3Y0YOoFPnjri6uQKQnpYuD+w/JJZ+vIvUKyX+3/kKsLAoIsr1r5Ci26jHi+a1uL4EYn7J/M0SkFISdyIOddVmuedEqjDa4T1MckTG4VWflPJSRcBkCfMA7xvHWjRw4dmJg4jo1QMXs0uJ8zMzM/n8s11y0YZvjJxxEnge2Kxrqiwl8jpIWAzc9xseJQlwDYWFheyPOsg7K7ZwJqfAyBPLgUW6piYVI/97AW9J6Fx80pQhEQwe0h+vRl7lCttTSadYu24zkTGGUX+Aq83Xz3RNddzwG5oKeFVevUr/1YnXM8iHyROHlS7ANVzKuMRn276Q727aZ+QJB/A1sFdAuoTRQI/iRgOtgYwePYSQ0JCbXru6rqMdjmWZGnmteVqiVsA24AeuCj8GcC8eec+MG0Dvvr0wm83lE+AaUs6k8NWur1m25QD5jvJXsUN7tGLgwH60a98WRalY8pmXl0dM9GH+9Z9dHDqVUe557XwaMGrY7+h5T3c8PT3LtwSMkJ2VRfzPCSQmJHE25QKFdjuNvDyxWDw4fz4Nh8OBj48XgYEtCAkNpknTJpV+ftvtdpKTTpGYcJKTJ0+RnZ2L2eyCv19jMjOzuZyTi7ubGy0D/AgKDiQwKAAXl9/uNbckwO0EE3c46gWoF+AOx/8A5/GCzwgOj0EAAAAASUVORK5CYII='
			alt="Logo" style="width: 90px" />
	</h3>
	<h2 style='text-align: center'>
		<br>dilbo installer / dilbo Installation
	</h2>

<?php

// Paths
// =====
$src_path = "https://dilbo.org/src/dilbo_server.zip";
$version_path = "https://dilbo.org/src/dilbo_version";
$mode_install = strcasecmp($_GET["do"], "install") == 0;
$version = file_get_contents($version_path);
echo "<h3 style='text-align: center'>app " . $version . "</h3>";

if (! $mode_install) {
    ?>
	<p style='text-align: center'>
		Most available settings and the database will not be modified.<br>The
		installation will start immediately after clicking the button<br> <br>Die
		meisten Einstellungen und Datenbank werden nicht verändert<br>Die
		Installation startet umgehend mit dem Klick auf die Schaltfläche
	</p>
	<p style='text-align: center'>
		<br> <a href='?do=install' class='formbutton'>install now</a>
	</p>
<?php
} else {

    // check loaded modules
    // ==============================================================================================
    $referenceModules = ["calendar", "Core", "ctype", "date", "exif", "fileinfo", "filter", "ftp", "gettext", "hash",
        "iconv", "json", "libxml", "mbstring", "mysqli", "openssl", "pcre", "pdo_mysql", "PDO", "Phar", "posix",
        "Reflection", "session", "SPL", "standard", "tokenizer", "zip", "zlib"
    ];
    $thisServerModules = get_loaded_extensions();
    $missing = [];
    foreach ($referenceModules as $referenceModule) {
        $contained = false;
        foreach ($thisServerModules as $thisServerModule)
            $contained = $contained || (strcmp($thisServerModule, $referenceModule) == 0);
        if (!$contained)
            $missing[] = $referenceModule;
    }
    echo "<p style='text-align: center'>Checking installed PHP-modules. / Installierte PHP-Module wurden geprüft.<br>";
    if (count($missing) > 0) {
        echo "These are missing: / Diese fehlen:<br>";
        foreach ($missing as $m)
            echo "'" . $m . "', ";
        echo "Maybe app runs without, but that was not tested. / Vielleicht läuft app auch ohne, aber es wurde nicht getestet.<br><br>";
    } else
        echo "OK.<br><br>";

    // fetch program source
    // ==============================================================================================
    echo "Getting source code from / Lade den Quellcode von: " . $src_path . " ...<br>";
    file_put_contents("src.zip", file_get_contents($src_path));
    echo " ... done / abgeschlossen. SIze / Größe: " . filesize("src.zip") . ".<br><br>";
    if (filesize("src.zip") < 1000) {
        echo "</p><p style='text-align: center'>Something went wrong. The file is too small. The download " .
            "apparently didn't work as expected. Aborting.</p></body></html>";
        echo "</p><p style='text-align: center'>Die Größe des Quellcode-Archivs ist zu klein. Da hat " .
            "etwas mit dem Download nicht geklappt. Deswegen bricht der Prozess hier ab.</p></body></html>";
        exit();
    }

    // read settings, will be used as cache
    // ==============================================================================================
    echo "<p style='text-align: center'>" . "Saving the existing configuration / Zwischenspeichern der bestehenden Konfiguration</p>";
    $pathsToSave = [ "Config/db", "Config/app", "Config/catalogs", "Config/lists", "Config/ui", "resources/app_colors.txt",
        "resources/clubLogo.png" ];
    $settings = [];
    foreach ($pathsToSave as $pathToSave)
        if (file_exists("../$pathToSave"))
            $settings[$pathToSave] = file_get_contents("../$pathToSave");

    // Open zip archive
    // ==============================================================================================
    echo "<p style='text-align: center'>Checking code archive / Überprüfe code-Archiv" . " ... ";
    $zip = new ZipArchive();
    $res = $zip->open('src.zip');
    if ($res !== TRUE) {
        echo "Opening the code archive failed. Aborting upgrade. / Das Öffnen des code-Archivs schlug fehl. Abbruch.</p>";
        unlink("src.zip");
        exit();
    }

    // Delete server side files.
    // ==============================================================================================
    echo "Deleting old code / Lösche alten code ...<br>";

    function rrmdir($dir): void
    {
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != "." && $object != "..") {
                    if (is_dir($dir . DIRECTORY_SEPARATOR . $object) && !is_link($dir . "/" . $object))
                        rrmdir($dir . DIRECTORY_SEPARATOR . $object);
                    else
                        unlink($dir . DIRECTORY_SEPARATOR . $object);
                }
            }
            rmdir($dir);
        }
    }

    // delete other code. Keep all produced application contents, including the resources. That will
    // nevertheless overwrite the files by those provided by the application package.
    $topLevelDirs = scandir(".");
    $directoriesToKeep = [ "Attachments", "Authentication", "Log", "Pdfs", "resources", "Run", "Uploads" ];
    foreach ($topLevelDirs as $topLevelDir) {
        $delete = str_starts_with($topLevelDir, "_") || str_starts_with($topLevelDir, "js")
            || ((strcmp(strtolower($topLevelDir), $topLevelDir) != 0) && !in_array($topLevelDir, $directoriesToKeep));
        if ($delete) {
            echo $topLevelDir . " ... ";
            rrmdir($topLevelDir);
        }
    }
    echo "<br>Done / Erledigt.<br>";

    // Unpack source files
    // ==============================================================================================
    echo "Unpacking source archive / Entpacke das Quellcode-Archiv ...<br>";
    $zip->extractTo('.');
    $zip->close();
    echo ' ... fertig. ... <br><br>';
    unlink("src.zip");

    // restore settings
    // ==============================================================================================
    echo "<p style='text-align: center'>restoring the existing configuration. / Stelle die bestehende Konfiguration wieder her.</p>";
    foreach ($settings as $pathToSave => $setting)
        file_put_contents("../$pathToSave", $setting);

    // Set directories' access rights.
    // ==============================================================================================
    echo "<p style='text-align: center'>Setting access rights / Setze die Zugriffsberechtigung ...<br>";
    $topLevelDirs = scandir(".");
    foreach ($topLevelDirs as $topLevelDir) {
        if (($topLevelDir != ".") && ($topLevelDir != "..")) {
            $isForbidden = (strcmp(strtolower($topLevelDir), $topLevelDir) != 0) && ($topLevelDir != "_Install");
            // keep the installation path open to continue with the process.
            if ($isForbidden) {
                chmod($topLevelDir, 0700);
                file_put_contents("$topLevelDir/.htaccess", "deny for all");
                echo "protecting " . $topLevelDir, ", ";
            } else {
                chmod($topLevelDir, 0755);
                echo "opening " . $topLevelDir, ", ";
                $contained = scandir($topLevelDir);
                foreach ($contained as $file)
                    chmod($topLevelDir . "/" . $file, 0755);
                unlink("$topLevelDir/.htaccess");
            }
        }
    }
    echo " ... Done. / Fertig.</p>";

?>
<p style='text-align: center'>
		Please add now the data base connection settings / Bitte jetzt die Datenbankverbindung eintragen <br><br>
        <a class='formButton' href='_Install/setup_db_connection.php'>Setup db</a>
	</p>
    <?php
}
    ?>

	<p style='text-align: center'>&nbsp;</p>
	<p style='text-align: center'>
		<small>&copy; dilbo.org</small>
	</p>
</body>
</html>
