<?php
/**
 * tools-for-your-hobby
 * https://www.tfyh.org
 * Copyright  2023-2025  Martin Glade
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

namespace tfyh\data;
include_once "../_Data/Codec.php";
include_once "../_Data/Formatter.php";
include_once "../_Data/Item.php";
include_once "../_Data/Parser.php";
include_once "../_Data/Property.php";
include_once "../_Data/PropertyName.php";
include_once "../_Data/Record.php";
include_once "../_Data/Type.php";
include_once "../_Data/ParserConstraints.php";

use tfyh\control\Logger;
use tfyh\control\LoggerSeverity;
include_once "../_Control/Logger.php";
include_once "../_Control/LoggerSeverity.php";

use tfyh\util\Language;
include_once "../_Util/Language.php";

use DateTimeZone;

const DEFAULT_TIME_ZONE = "Europe/Berlin";

/**
 * A utility class to load all application configuration.
 */
class Config
{
    public static array $allSettingsFiles = [
        // the sequence is relevant for the loading process, do not change
        // settings descriptor
        "descriptor", // the file to define the properties which describe a value
        "types", // the file to define the available value types
        // immutable settings. These will never change in structure nor get actual values
        "access", // the settings which configure roles, menus, workflows, concessions and subscriptions
        "templates", // configuration branches which are available for multiple use in this app
        "framework", // the setting which configure the framework classes for this app
        "tables", // the database table layout
        // tenant mutable structure. These may get added or deleted items and actual values
        "lists", // the settings which configure list retrievals off the database
        "app", // all settings of the application needed to run at the tenant.
        "club", // all settings of the tenant club as organisation.
        "catalogs", // the catalogs of types, like the valid boat variants asf.
        "ui", // user interface layout and profile settings
        "theme" // user interface design theme
    ];

    public static array $allSettingsDirs = [
        // the sequence is relevant for the loading process, do not change
        "basic", // this is provided as part of the release distribution
        "added", // all structure additions for the tenant, including tenant specific UI profiles
    ];

    private static array $rootItemDefinition = [ "_path" => "#none", "_name" => "root", "default_label" => "root", "value_type" => "none" ];
    private static array $invalidItemDefinition = [ "_path" => "#none", "_name" => "invalid", "default_label" => "invalid item", "value_type" => "none" ];

    private static Config $instance;
    static function getInstance(): Config {
        if (!isset(self::$instance))
            // create the instance
            self::$instance = new self();
        return self::$instance;
    }

    /**
     * Return the modification times of the configuration files for the settings loader of the client or the
     * Configuration panel.
     */
    public static function getModified(): String {
        $modified = "";
        foreach (self::$allSettingsDirs as $settingsDir) {
            foreach (self::$allSettingsFiles as $settingsFile) {
                $fName = "Config/$settingsDir/$settingsFile";
                if (file_exists("../$fName"))
                    $modified .= "\n" . $fName . "=" . filemtime("../$fName");
            }
        }
        return substr($modified, 1);
    }

    // The root and invalid item. They have no configuration file entry. They will be initialized within config->load()
    // Due to PHP language limitations.
    public Item $invalidItem;
    public Item $rootItem;
    private array $settingsCsv = [];
    private array $loaded = [];
    private Language $language = Language::DE;
    private DateTimeZone $timeZone;
    public Logger $logger;
    public String $appName = "";
    public string $appVersion = "0.0";
    public String $appUrl = "";

    private function __construct() {
        // while monitor and runner share the same session type logger, the configuration errors and warnings
        // get a different one, because they will reissue the same warnings on every page.
        $this->logger = new Logger("../Log/config.log");
    }

    /**
     * Get an Item by its path. Returns the invalid Handle on errors
     */
    public function getItem(string $path): Item
    {
        if (strlen($path) == 0)
            return self::$instance->rootItem;
        if (!str_starts_with($path, "."))
            return self::$instance->invalidItem;
        $names = explode(".", substr($path, 1));
        $i = 0;
        $parent = $this->rootItem;
        while (($i < count($names) && !is_null($parent) && $parent->hasChild($names[$i])))
            $parent = $parent->getChild($names[$i++]);
        if ($parent != null) {
            if ($i == count($names))
                return $parent; // path fully resolved
            if ($parent === $parent->parent()) {
                // hit top level
                if (!($this->loaded[$names[0]] ?? false)) {
                    $this->loadBranch($names[0]);
                    $this->logger->log(LoggerSeverity::INFO, "Config->getItem()", "Path '"
                        . $names[0] . "' loaded.");
                    return $this->getItem($path);
                } else {
                    $this->logger->log(LoggerSeverity::ERROR, "Config->getItem()", "Path '$path' not found");
                    return self::$instance->invalidItem; // path not resolved
                }
            }
        }
        $this->logger->log(LoggerSeverity::ERROR, "Config->getItem()", "Path '$path' not found");
        return self::$instance->invalidItem; // path not resolved
    }

    public function language(): Language { return $this->language; }
    public function timeZone(): DateTimeZone {
        if (!isset($this->timeZone))
            $this->timeZone = new DateTimeZone(DEFAULT_TIME_ZONE);
        return $this->timeZone;
    }

    /**
     * load a top level branch. This is not part of the main loading procedure, but performed on demand based on
     * the getItem() requests.
     */
    private function loadBranch($branchName): void {
        if ($this->loaded[$branchName] ?? false)
            return;

        foreach (Config::$allSettingsDirs as $settingsDir) {
            $isBasic = ($settingsDir == "basic");
            if (isset($this->settingsCsv[$settingsDir][$branchName])) {
                $settingsMap = Codec::csvToMap($this->settingsCsv[$settingsDir][$branchName]);
                $loadingResult = $this->rootItem->readBranch($settingsMap, $isBasic);
                if (strlen($loadingResult) > 0)
                    $this->logger->log(LoggerSeverity::ERROR, "Config->load",
                        "[$settingsDir]: $loadingResult");
            }
        }
        $this->loaded[$branchName] = true;
    }
    /**
     * Load the configuration. This will only parse CSV files, no language setting needed beforehand.
     */
    public function load(): void
    {
        // only for the PHP, because the class initializer cannot handle expressions
        ParserConstraints::init();

        // initialise the Type object
        $descriptorCsv = file_get_contents("../Config/basic/descriptor");
        $typesCsv = file_get_contents("../Config/basic/types");
        Type::init($descriptorCsv, $typesCsv);

        // Initialize the root and invalid item. They have no configuration file entry.
        // PHP only due to initialization language restrictions. The nullish operator ensures rootItem and invalidItem
        // to stay the same, if reloading occurs, e.g. in the _pages/error.php page
        $this->rootItem = $this->rootItem ?? Item::getFloating(self::$rootItemDefinition);
        $this->invalidItem = $this->invalidItem ?? Item::getFloating(self::$invalidItemDefinition);

        // read the settings
        foreach (Config::$allSettingsDirs as $settingsDir) {
            $this->settingsCsv[$settingsDir] = [];
            foreach (Config::$allSettingsFiles as $settingsFile)
                if (file_exists("../Config/$settingsDir/$settingsFile"))
                    $this->settingsCsv[$settingsDir][$settingsFile] = file_get_contents("../Config/$settingsDir/$settingsFile");
        }

        // set tables and language
        $loadTime = microtime(true);
        Record::copyCommonFields();
        file_put_contents("../Log/tablesInitTimes.log", strval(microtime(true) - $loadTime) . "\n", FILE_APPEND);
        $languageString = $this->getItem(".app.user_preferences.language")->valueCsv();
        $this->language = Language::valueOfOrDefault(strtoupper($languageString));
        $this->appName = $this->getItem(".framework.app.name")->valueCsv();
        $this->appUrl = $this->getItem(".framework.app.url")->valueCsv();
        $this->appVersion = file_get_contents("../public/version");

        // initialize the locale settings for parser and formatter (JavaScript code: see main loader.)
        $this->logger->setLocale($this->language);
        Formatter::setLocale($this->language());
        Parser::setLocale($this->timeZone());
    }
}