<?php
/**
 * tools-for-your-hobby
 * https://www.tfyh.org
 * Copyright  2023-2025  Martin Glade
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

namespace tfyh\data;
include_once "../_Data/Codec.php";
include_once "../_Data/Formatter.php";
include_once "../_Data/ParserName.php";

use DateTimeImmutable;
use tfyh\util\I18n;

class Type
{

    // The list of types available
    public static array $types = array();
    public static Type $invalid;

    /**
     * Initialize the descriptor and types
     */
    public static function init(string $descriptorCsv, string $typesCsv): void
    {
        // initialize descriptor
        $definitionsArray = Codec::csvToMap($descriptorCsv);
        Property::$descriptor = [];
        Property::$invalid = new Property(Property::$invalidPropertyDefinition); // no static initialization possible
        Property::$descriptor[PropertyName::INVALID->name] = Property::$invalid;
        foreach ($definitionsArray as $definition)
            Property::$descriptor[PropertyName::valueOfOrInvalid($definition["name"])->value] = new Property($definition);
        // initialize types catalog
        $definitionsArray = Codec::csvToMap($typesCsv);
        Type::$types = [];
        foreach ($definitionsArray as $definition)
            Type::$types[$definition["_name"] ?? "Oops! No name."] = new Type($definition);
        self::$invalid = new Type([ "_name" => "invalid", "default_label" => "invalid", "parser" => "none"]);
    }

    /**
     * Return the requested type. If there is no match, return Type.invalidType, which is
     * created here during bootstrap.
     */
    public static function get(string $typeName) { return self::$types[$typeName] ?: Type::$invalid; }

    // the type properties define the content of the value, e.g. its limits, its parser, its form input
    // and SQL representation asf.
    public string $name;
    public ParserName $parser;
    private array $properties;

    public function __construct(array $definition)
    {
        $this->name = isset($definition["_name"]) ? strval($definition['_name']) : "missing_type_name";
        $this->parser = isset($definition["parser"]) ? ParserName::valueOfOrNone($definition['parser']) : ParserName::NONE;
        // set the properties. Parse them first and ensure the immutable properties are set.
        $this->properties = Property::parseProperties($definition, $this);
        $this->properties[PropertyName::_NAME->value] = $this->name;
        $this->properties[PropertyName::_PATH->value] = "";
        $this->properties[PropertyName::VALUE_TYPE->value] = $this->name;
    }

    public function name(): String { return $this->name; }
    public function label(): String { return $this->defaultLabel(); }
    public function description(): String { return $this->defaultDescription(); }

    public function parser(): ParserName {
        return $this->parser;
    }
    public function defaultValue(): bool|int|float|DateTimeImmutable|string|array {
        return $this->properties[PropertyName::DEFAULT_VALUE->value] ?? ParserConstraints::empty($this->parser);
    }
    // localized, i.e. translated properties
    public function defaultLabel(): string {
        return I18n::getInstance()->t(strval($this->properties[PropertyName::DEFAULT_LABEL->value] ?? ""));
    }
    public function defaultDescription(): string {
        return I18n::getInstance()->t(strval($this->properties[PropertyName::DEFAULT_DESCRIPTION->value] ?? ""));
    }
    // property getter functions
    public function nodeHandling(): string { return strval($this->properties[PropertyName::NODE_HANDLING->value] ?? ""); }
    public function nodeAddableType(): string { return strval($this->properties[PropertyName::NODE_ADDABLE_TYPE->value] ?? ""); }
    public function nodeWritePermissions(): string { return strval($this->properties[PropertyName::NODE_WRITE_PERMISSIONS->value] ?? ""); }
    public function nodeReadPermissions(): string { return strval($this->properties[PropertyName::NODE_READ_PERMISSIONS->value] ?? ""); }

    // Types are immutable and do not have actual values.
    public function valueMin(): bool|int|float|DateTimeImmutable|string|array {
        return $this->properties[PropertyName::VALUE_MIN->value] ?? ParserConstraints::min($this->parser);
    }
    public function valueMax(): bool|int|float|DateTimeImmutable|string|array {
        return $this->properties[PropertyName::VALUE_MAX->value] ?? ParserConstraints::max($this->parser);
    }
    public function valueSize(): int { return intval($this->properties[PropertyName::VALUE_SIZE->value] ?? "0"); }
    public function valueUnit(): string { return $this->properties[PropertyName::VALUE_UNIT->value] ?? ""; }
    public function valueReference(): string { return $this->properties[PropertyName::VALUE_REFERENCE->value] ?? ""; }
    public function validationRules(): string { return $this->properties[PropertyName::VALIDATION_RULES->value] ?? ""; }
    public function sqlType(): string { return $this->properties[PropertyName::SQL_TYPE->value] ?? ""; }
    public function sqlNull(): bool { return (bool) $this->properties[PropertyName::SQL_NULL->value]; }
    public function sqlIndexed(): string { return $this->properties[PropertyName::SQL_INDEXED->value] ?? ""; }
    public function inputType(): string { return $this->properties[PropertyName::INPUT_TYPE->value] ?? "text"; }
    public function inputModifier(): string { return $this->properties[PropertyName::INPUT_MODIFIER->value] ?? ""; }
    public function recordEditForm(): string { return $this->properties[PropertyName::RECORD_EDIT_FORM->value] ?? ""; }

    public function __toString() { return $this->name; }

}
