<?php

namespace tfyh\util;

use tfyh\control\Runner;
include_once "../_Control/Runner.php";

use JetBrains\PhpStorm\NoReturn;
use ZipArchive;

class FileHandler
{
    /**
     * Parse a file system tree and return all relative path names of files. Runs recursively.
     */
    private static function listFilesOfTree (array $filePaths, String $branchRootDir, String $parentDir): array
    {
        $handle = opendir($branchRootDir . $parentDir);
        if ($handle !== false) {
            while (false !== ($entry = readdir($handle))) {
                if ($entry != "." && $entry != "..") {
                    // add relative path
                    $filePaths[] = $parentDir . $entry;
                    // drill down, if this is a directory
                    if (is_dir($branchRootDir . $parentDir . $entry)) {
                        $filePaths = self::listFilesOfTree($filePaths, $branchRootDir,
                            $parentDir . $entry . "/");
                    }
                }
            }
            closedir($handle);
            return $filePaths;
        }
        return [];
    }

    /**
     * Parse a file system branch and return all relative path names of files
     */
    public static function listFilesOfBranch (String $branchRootDir): array
    {
        $filePaths = [];
        return self::listFilesOfTree($filePaths, $branchRootDir, "");
    }

    /**
     * See
     * https://stackoverflow.com/questions/3338123/how-do-i-recursively-delete-a-directory-and-its-entire-contents-files-sub-dir
     */
    public static function rrmdir (String $dir, bool $echo = false): bool
    {
        $success = false;
        if (is_dir($dir)) {
            $success = true;
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != "." && $object != "..") {
                    if (is_dir($dir . DIRECTORY_SEPARATOR . $object) && ! is_link($dir . "/" . $object)) {
                        if ($echo)
                            echo "drill down into " . $dir . DIRECTORY_SEPARATOR . $object . "<br>";
                        $success &= self::rrmdir($dir . DIRECTORY_SEPARATOR . $object);
                    } else {
                        if ($echo)
                            echo "unlinking " . $dir . DIRECTORY_SEPARATOR . $object . "<br>";
                        $success &= unlink($dir . DIRECTORY_SEPARATOR . $object);
                    }
                    if (! $success && $echo)
                        echo "failed.<br>";
                }
            }
            $success &= rmdir($dir);
            if (! $success && $echo)
                echo "failed to remove $dir.<br>";
        }
        return $success;
    }

    /**
     * Unzip for table import. See https://www.php.net/manual/de/ref.zip.php
     */
    public static function unzip (String $zipPath, bool $rmdir = false): array|string
    {
        $i18n = I18n::getInstance();
        $dirPath = substr($zipPath, 0, strrpos($zipPath, "."));
        if (! file_exists($zipPath))
            return $i18n->t("XFt9AM|#Error: Zip path °%1° do...", $zipPath);

        if (file_exists($dirPath)) {
            if ($rmdir) {
                $removed = self::rrmdir($dirPath);
                if (! $removed)
                    return $i18n->t("5qjixH|#Error: Target directory...", $dirPath);
            } else
                return $i18n->t("5qjixH|#Error: Target directory...", $dirPath);
        }
        mkdir($dirPath);

        $zip = new ZipArchive();
        $resource = $zip->open($zipPath);
        if (! $resource || is_numeric($resource))
            return $i18n->t("6qoA8U|#Error while opening the...", $zipPath);

        $zip->extractTo($dirPath);
        $fileList = [];
        for ($i = 0; $i < $zip->numFiles; $i++)
            $fileList[] = $zip->getNameIndex($i);
        $zip->close();
        return $fileList;
    }

    /**
     * Store a set of files into a given archive.
     */
    public static function zip_files (array $srcFilePaths, String $zipFilepath): void
    {
        $zip = new ZipArchive();
        if ($zip->open($zipFilepath, ZipArchive::CREATE) !== true) {
            file_put_contents($zipFilepath, "");
        }
        foreach ($srcFilePaths as $src_filepath)
            if (! is_dir($src_filepath))
                if (file_exists($src_filepath))
                    $zip->addFile($src_filepath);
        $zip->close();
    }

    /**
     * Return a file to the user. Uses the "header" function, i.e. must be called before any other output is
     * generated by the calling page. The file is kept, its content type iis decoded, on failure
     * "application/x-binary" is used.
     *
     * @param String $filepath
     *            path to file which shall be returned.
     */
    #[NoReturn] public static function return_file_to_user (String $filepath): void
    {
        // return file.
        $filename = (str_contains($filepath, "/")) ? substr($filepath, strrpos($filepath, "/") + 1) : $filepath;
        $mime_content_type = mime_content_type($filepath);
        if ($mime_content_type === false)
            $mime_content_type = "application/x-binary";
        if (file_exists($filepath)) {
            header($_SERVER["SERVER_PROTOCOL"] . " 200 OK");
            header("Cache-Control: public"); // needed for Internet Explorer
            header("Content-Type: " . $mime_content_type);
            header("Content-Transfer-Encoding: Binary");
            header("Content-Length:" . filesize($filepath));
            header("Content-Disposition: attachment; filename=" . $filename);
            readfile($filepath);
            // unlink($filepath); That results in an execution error. Remove the file in housekeeping.
            Runner::getInstance()->endScript(false);
        } else {
            die(I18n::getInstance()->t("8XihSu|Error: File °%1° not fou...", $filepath));
        }
    }

    /**
     * Return a file to the user. The file will afterward not be deleted (unlinked). Uses the "header"
     * function, i.e. must be called before any other output is generated by the calling page.
     */
    #[NoReturn] private static function return_zip_file (String $filepath): void
    {
        // return zip.
        if (file_exists($filepath)) {
            header($_SERVER["SERVER_PROTOCOL"] . " 200 OK");
            header("Cache-Control: public"); // needed for Internet Explorer
            header("Content-Type: application/zip");
            header("Content-Transfer-Encoding: Binary");
            header("Content-Length:" . filesize($filepath));
            header("Content-Disposition: attachment; filename=" . $filepath);
            readfile($filepath);
            // unlink($filepath); That results in an execution error. Remove the file in housekeeping.
            exit(); // really exit. No test case left over.
        } else {
            die(I18n::getInstance()->t("6HrzgB|Error: File °%1° not fou...", $filepath));
        }
    }

    /**
     * Return it files in a compressed archive to the user. Uses the "header" function, i.e. must be called
     * before any other output is generated by the calling page.
     */
    #[NoReturn] public static function returnFilesAsZip (array $srcFilePaths, String $zipFilename, bool $removeFiles): void
    {
        self::zip_files($srcFilePaths, $zipFilename);
        if ($removeFiles) {
            foreach ($srcFilePaths as $src_filepath) {
                unlink($src_filepath);
            }
        }
        self::return_zip_file($zipFilename);
    }

    /**
     * Store a String into the given filepath and create a zip archive at the $filepath . ".zip".
     */
    public static function zip ($string_to_zip, $filename): string
    {
        $zip = new ZipArchive();
        $zipFilename = $filename . ".zip";

        if ($zip->open($zipFilename, ZipArchive::CREATE) !== TRUE) {
            exit("cannot open <$zipFilename>\n"); // no i18n required
        }
        if ($zip->addFromString($filename, $string_to_zip) !== true)
            exit("cannot write zip <$zipFilename>\n"); // no i18n required
        $zip->close();
        return $zipFilename;
    }

    /**
     * Zip a csv-String and return it as file to the user. Uses the "header" function, i.e. must be called
     * before any other output is generated by the calling page.
     */
    #[NoReturn] public static function returnStringAsZip (String $string, String $fName): void
    {
        $zipFilename = self::zip($string, $fName);
        self::return_zip_file($zipFilename);
    }

    /**
     * Scan a directory and return the contents as HTML table
     */
    public static function get_dir_contents (String $dir, int $level_of_top = 1): string
    {
        $i18n = I18n::getInstance();
        $result = "<table>";
        $result .= "<tr class=flist><td>&nbsp;</td><td>" . $dir . "</td><td>" . $i18n->t("F7t6Jn|Action") .
            "</td></tr>";
        $items = 0;
        $cdir = scandir($dir);
        if ($cdir)
            foreach ($cdir as $value) {
                if (! in_array($value, array(".",".."
                ))) {
                    if (is_dir($dir . DIRECTORY_SEPARATOR . $value)) {
                        $result .= "<tr class=flist><td><img alt='folder' src='../resources/drive_folder-20px.png' title='" .
                            $i18n->t("hX0dDX|Directory") . "' /></td>" . "<td><a href='?cdir=" . $dir . "/" .
                            $value . "'>" . $value . "</a>&nbsp;&nbsp;&nbsp;&nbsp;</td>" .
                            "<td><a href='?xdir=" . $dir . "/" . $value .
                            "'><img alt='delete' src='../resources/delete_file-20px.png' title='" .
                            $i18n->t("MyjroH|delete directory, if emp...") . "' /></a>" . "</td></tr>\n";
                    } else {
                        $result .= "<tr class=flist><td><img alt='file' src='../resources/drive_file-20px.png' title='" .
                            $i18n->t("wdLerX|File") . "' /></td>" . "<td>" . $value .
                            "&nbsp;&nbsp;&nbsp;&nbsp;</td><td><a href='?dfile=" . $dir . "/" . $value .
                            "'><img alt='download' src='../resources/download_file-20px.png' title='" .
                            $i18n->t("cmjKme|Download file") . "' /></a>" . "<a href='?xfile=" . $dir . "/" .
                            $value . "'><img alt='delete' src='../resources/delete_file-20px.png' title='" .
                            $i18n->t("c32XmM|Delete file") . "' /></a>" . "</td></tr>\n";
                    }
                    $items ++;
                }
            }
        if ($items == 0)
            $result .= "<tr class=flist><td>" . $i18n->t("HpsAcF|(empty)") . "</td><td>" .
                $i18n->t("26WMwL|no content found.") . "</td></tr>";
        $parentDir = (strrpos($dir, "/") > 0) ? substr($dir, 0, strrpos($dir, "/")) : $dir;
        // the topmost offered parent directory is the "uploads" folder to ensure
        // entry into the application files hierarchy is not possible.
        if (count(explode("/", $parentDir)) > $level_of_top)
            $result .= "<tr class=flist><td><img alt='file' src='../resources/drive_file-20px.png' title='" .
                $i18n->t("CBxZVW|One level higher") . "' /></td><td><a href='?cdir=" . $parentDir . "'>" .
                $parentDir . "</a></td></tr>";
        $result .= "</table>";

        return $result;
    }

}