<?php

namespace tfyh\util;
include_once "../_Util/ListHandlerKernel.php";

use DateTimeImmutable;
use JetBrains\PhpStorm\NoReturn;

use tfyh\control\Runner;
use tfyh\control\Sessions;
include_once "../_Control/Runner.php";
include_once "../_Control/Sessions.php";

use tfyh\data\Config;
use tfyh\data\DatabaseConnector;
use tfyh\data\Formatter;
use tfyh\data\Parser;
use tfyh\data\ParserConstraints;
use tfyh\data\ParserName;
use tfyh\data\Validator;
include_once "../_Data/Config.php";
include_once "../_Data/Formatter.php";
include_once "../_Data/ParserName.php";
include_once "../_Data/Validator.php";

/**
 * This class provides a list segment for a web file. <p>The definition must be a CSV-file, all entries
 * without line breaks, with the first line being always "id;permission;name;select;from;where;options" and
 * the following lines the respective values. The values in select, from, where and options are combined to
 * create the needed SQL-statement to retrieve the list elements from the database.</p><p>options
 * are<ul><li>sort=[-]column[.[-]column]: order by the respective column in ascending or descending (-)
 * order</li><li>filter=column.value: filter the column for the given value, always using the LIKE operator
 * with '*' before and after the value</li><li>link=[link]: link the first column to the given url e.g.
 * '../_forms/changeUser.php?id=id' replacing the column name at the end (here: id) by the respective
 * value.</li></ul></p> <p>The list is always displayed as a table grid. It will show the default sorting, if
 * no sorting option is provided.</p>
 */
class ListHandler extends ListHandlerKernel
{

    /**
     * Limit the size on an entry to a number of characters
     */
    public int $entrySizeLimit = 0;

    /**
     * Build a list set based on the definition provided in the csv file at "../Config/lists/$set". Use the list with
     * name $nameOrDefinition as current list name or none, if $name = "", or put your full set definition to
     * $nameOrDefinition and "@dynamic" to $set to generate a list programmatically. Use the count() function to see
     * whether list definitions could be parsed.
     */
    public function __construct(string $set, string $nameOrDefinition = "", array $args = []) {
        parent::__construct($set, $nameOrDefinition, $args);
    }

    /**
     * Get the arguments used in a list definition as comma separated string.
     */
    public function getArgs (array $list_definition): string
    {
        $args = "";
        foreach ($list_definition as $value) {
            $brace_open = - 1;
            while ($brace_open !== false) {
                $brace_open = strpos($value, "{", $brace_open + 1);
                $brace_close = ($brace_open === false) ? false : strpos($value, "}", $brace_open);
                if (($brace_close !== false) && ($brace_open < $brace_close))
                    $args .= "," . substr($value, $brace_open + 1, $brace_close - $brace_open - 1);
            }
        }
        if (strlen($args) > 0)
            return substr($args, 1);
        return "";
    }

    /**
     * Return a zip-Download-link for this list based on its definition or the provided options.
     */
    public function getLink(string $oSortsList, string $oFilter, string $oFValue, int $zipMode): string
    {
        if ($this->noValidCurrentList())
            return "?set=" . $this->set . "&name=";
        $sortString = (strlen($oSortsList) == 0) ? "" : "&sort=" . $oSortsList;
        $filterString = (strlen($oFilter) == 0) ? "" : "&filter=" . $oFilter;
        $fValueString = (strlen($oFValue) == 0) ? "" : "&fvalue=" . $oFValue;
        return "?set=" . $this->set . "&name=" . $this->listDefinitions[$this->currentListIndex]["name"] .
            "&zip=" . $zipMode . $sortString . $filterString . $fValueString;
    }

    /**
     * Return a html code of this list based on its definition or the provided options.
     */
    public function getHtml(string $oSortsList, string $oFilter, string $oFValue, array $pivot): string
    {
        if ($this->noValidCurrentList())
            return "<p>" . $this->i18n->t("4t2ytU|Application configuratio...") . "</p>";

        $maxRows = 100;
        $rowsReferenced = $this->getRows("referenced", $oSortsList, $oFilter, $oFValue, $maxRows + 1);
        $errorMessage = "";
        if (!is_array($rowsReferenced)) {
            $errorMessage .= $rowsReferenced;
            $rowsReferenced = [ $rowsReferenced ];
        }
        $countOfListRows = $this->rowsSqlCount;
        $rowsFound = ($countOfListRows > 0);
        $retrievalError = DatabaseConnector::getInstance()->getError();
        $errorMessage .= (strlen($retrievalError) == 0)
            ? "" : I18n::getInstance()->t("cN3sAX|Data retrieval error:" . DatabaseConnector::getInstance()->getError());
        $rowsToShow = (($countOfListRows > $maxRows) ? "&gt;&nbsp;$maxRows" : "$countOfListRows");
        $countNotice = ($rowsFound)
                ? "$rowsToShow   " . $this->i18n->t("C94hEq|Records found.")
                : $this->i18n->t("cwpbfK|The list is empty or no ...") . " " . $errorMessage;
        $countNotice = "<b>" . $countNotice . "</b>";
        if (strlen($errorMessage) > 0)
            $countNotice = "<h5>" . $errorMessage . "</h5>";

        // Find result and usage explanation
        $outHtml = "<p>" . $countNotice;
        if ($countOfListRows > 0)
            $outHtml .= " " . $this->i18n->t("nLel3k|Sort with one click on t...");
        $outHtml .= "</p>";

        // table container
        $tableContainer = "<div style='overflow-x: auto; margin-top:12px; margin-bottom:10px;'>";
        $tableContainer .= "<table style='border: 2px solid transparent;'>";

        // table header
        $tableHeader = "<thead><tr>";
        foreach ($this->columns as $column) {
            // do not display technical ids
            if (($column != "uid") && ($column != "uuid")) {
                // identify ID-column for change link
                $cText = ($this->recordItem->hasChild($column)) ? $this->recordItem->getChild($column)->label() : $column;
                $sortsSplit = explode(".", $oSortsList);
                $isAscendingSortColumn = in_array($column, $sortsSplit);
                $isDescendingSortColumn = in_array("-" . $column, $sortsSplit);
                if ($isAscendingSortColumn || $isDescendingSortColumn) {
                    $cText = ($isDescendingSortColumn) ? $cText . '<br /><b>&nbsp;&nbsp;&#9650;</b> ' : $cText .
                        '<br /><b>&nbsp;&nbsp;&#9660;</b> ';
                    $cSort = ($isDescendingSortColumn) ? $column : '-' . $column;
                } else
                    $cSort = $column;
                $ofString = (strlen($oFilter) > 0) ? "&filter=" . $oFilter . "&fvalue=" . $oFValue : "";
                $listParameterStr = (isset($_GET["listparameter"])) ? "&listparameter=" . $_GET["listparameter"] : "";
                $tableHeader .= "<th><a class='table-header' href='?set=" . $this->set . "&name=" . $this->name .
                    "&sort=" . $cSort . $ofString . $listParameterStr . "'>" . $cText . "</a></th>";
            }
        }
        $tableHeader .= "</tr></thead>";

        // table body
        $tableBody = "<tbody>";
        for ($i = 0; ($i < $maxRows) && ($i < $countOfListRows); $i++) {
            $rowReferenced = $rowsReferenced[$i];
            $uid = $rowReferenced["uid"] ?? "";
            $rowHtml = "<tr>";
            $c = 0;
            foreach ($rowReferenced as $column => $valueReferenced)
                // do not display technical ids
                if (($column != "uid") && ($column != "uuid")) {
                    if ((strlen($uid) > 0) && ($c == 0))
                        $rowHtml .= "<td><b><a href='../_pages/viewRecord.php?table=" . $this->tableName .
                                "&uid=" . $uid . "'>" . $valueReferenced . "</a></b></td>";
                    else
                        $rowHtml .= "<td>" . $valueReferenced . "</td>";
                    $c++;
                }
            $rowHtml .= "</tr>\n";
            $tableBody .= $rowHtml;
        }
        $tableBody .= "</tbody>";

        // add table to HTML output
        if ($rowsFound)
            $outHtml .= $tableContainer . $tableHeader . $tableBody. "</table></div>";

        // capping notice and filter form
        if ($countOfListRows > 0) {
            $filterForm = "<p>" . $this->i18n->t("ZmqZlm|List capped after %1 rec...", $maxRows)
                . " " . $this->i18n->t("1ci7d1|Please use the filter or...") . "</p>";
            // filter form
            $filterLink = $this->getLink($oSortsList, $oFilter, $oFValue, 0);
            $filterForm .= "<form action='" . $filterLink . "'>" . $this->i18n->t("Rhz5mZ|Filter in column:") .
                "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
            $filterForm .= "<input type='hidden' name='name' value='" . $this->name . "' />";
            $filterForm .= "<input type='hidden' name='set' value='" . $this->set . "' />";
            if (strlen($oSortsList) > 0)
                $filterForm .= "<input type='hidden' name='sort' value='" . $oSortsList . "' />";
            $filterForm .= "<select name='filter' class='formSelector' style='width:20em'>";
            foreach ($this->columns as $column) {
                if (!str_contains($column, ".")) {
                    if (strcasecmp($column, $oFilter) == 0)
                        $filterForm .= '<option value="' . $column . '" selected>' . $column . "</option>\n";
                    else
                        $filterForm .= '<option value="' . $column . '">' . $column . "</option>\n";
                }
            }
            $filterForm .= "</select>";
            $filterForm .= "<br>" . $this->i18n->t("afDbIr|Value") .
                " <input type='text' name='fvalue' class='formInput' value='" . $oFValue .
                "'  style='width:19em' />" . "&nbsp;&nbsp;&nbsp;&nbsp;<input type='submit' value='" .
                $this->i18n->t("efjxwi|show filtered list") . "' class='formButton'/></form>";

            $outHtml .= $filterForm;
        }

        // list download link
        $zipLink = $this->getLink($oSortsList, $oFilter, $oFValue, 1);
        $outHtml .= "<p>" . $this->i18n->t("OrvMhQ|get as csv-download file...") . " <a href='" .
            $zipLink . "'>" . $this->tableName . ".zip</a>";
        $outHtml .= ". " . $this->i18n->t("TxBnFe|PLEASE NOTE: Use the inf...") . "</p>";

        // pivot table
        $pivotTableHtml = "";
        if (count($pivot) == 4) {
            $pivotTableHtml .=  "<h4>" . $this->i18n->t("FalTMo|Overview") . "</h4>";
            // pivot table download link
            $pivotZipLink = $this->getLink($oSortsList, $oFilter, $oFValue, 1);
            $pivotTableHtml .=  "<p>" . $this->i18n->t("OrvMhQ|get as csv-download file...") . " <a href='" .
                $pivotZipLink . "'>" . $this->tableName . ".pivot.zip</a>";
            $pivotTableHtml .= ". " . $this->i18n->t("TxBnFe|PLEASE NOTE: Use the inf...") . "</p>";
            // pivot table
            $pivotTable = new PivotTable($this, $pivot[0], $pivot[1], $pivot[2], $pivot[3]);
            $pivotTableHtml .= $pivotTable->getHtml();
        }
        $outHtml .= $pivotTableHtml;

        return $outHtml;
    }

    /**
     * Check whether a field is contained in a list.
     */
    public function hasField(string $fieldName): bool|int
    {
        return array_key_exists($fieldName, $this->columns);
    }

    /**
     * Provide a csv file, filter and sorting according to default. csv data content is UTF-8 encoded - i.e.
     * uses the data as they are provided by the database.
     */
    public function getCsv(string $oSortsList, string $oFilter, string $oFValue, String $onlyFirstOf = null): bool|string
    {
        if ($this->noValidCurrentList())
            return false;

        // format and compile list
        $csv = "";
        $records = $this->getRows("csv", $oSortsList, $oFilter, $oFValue);
        $lastChecked = null;
        $firstOf = "";
        $useEntrySizeLimit = ($this->entrySizeLimit > 0);
        $entrySizeLimit = ($this->entrySizeLimit < 10) ? 10 : $this->entrySizeLimit;
        $r = 0;
        foreach ($records as $record) {
            if ($r == 0) {
                foreach ($record as $key => $value)
                    $csv .= ";" . $key;
                $csv = substr($csv, 1) . "\n";
            }
            $rowStr = "";
            foreach ($record as $key => $valueCsv) {
                if ($useEntrySizeLimit && (strlen($valueCsv) > $entrySizeLimit))
                    $valueCsv = substr($valueCsv, 0, $entrySizeLimit - 3) . "...";
                if ((str_contains($valueCsv, "\"")) || (str_contains($valueCsv, "\n")) ||
                    (str_contains($valueCsv, ";")))
                    $rowStr .= '"' . str_replace('"', '""', $valueCsv) . '";';
                else
                    $rowStr .= ";" . $valueCsv;
                if (!is_null($onlyFirstOf) && ($key == $onlyFirstOf))
                    $firstOf = $valueCsv;
            }
            $r++;

            // add line or filter, if requested
            if (!$onlyFirstOf || is_null($lastChecked) || Validator::isEqualValues($firstOf, $lastChecked)) {
                $csv .= substr($rowStr, 1) . "\n";
                $lastChecked = $firstOf;
            }
        }
        return $csv;
    }

    /**
     * Provide a csv file for download. Will not return, but exit via $toolbox->return_string_as_zip()
     * function.
     */
    #[NoReturn] public function returnZip(string $oSortsList, string $oFilter, string $oFValue)
    {
        if ($this->noValidCurrentList())
            return "<p>" . $this->i18n->t("eG3R6d|Application configuratio...") . "</p>";

        $csv = $this->getCsv($oSortsList, $oFilter, $oFValue);
        // add timestamp, source and destination
        $sessions = Sessions::getInstance();
        $destination = $sessions->userFullName() . " (" . $sessions->userId() . ", " . $sessions->userRole() . ")";
        $csv .= "\n" . $this->i18n->t("JX2kP6|Provided on %1 by %2 to ...",
                Formatter::format(new DateTimeImmutable("now"), ParserName::DATETIME),
                $_SERVER['HTTP_HOST'], $destination) . "\n";
        FileHandler::returnStringAsZip($csv, $this->tableName . ".csv");
    }
}
