<?php

namespace tfyh\util;

use tfyh\control\LoggerSeverity;
use tfyh\control\Runner;
use tfyh\data\Ids;
use const tfyh\data\base64charsPlus;

/**
 * A utility class to create one-time tokens for user identification.
 */

class TokenHandler
{

    private const obfuscator = "jtzOjk6IjEyNy4wLjAuMSI7czoxMToiZGJfYWNjb3VudHMiO2E6MTp7czo0OiJyb290IjtzOjg6IlNmeDFubHAuIjt9czo3OiJkYl9uYW1lIjtzOjU6ImZ2c3NiIjtzO";

    /**
     * Obfuscate a base64 String by xor-operation with an obfuscating String. Apply the same procedure to restore it.
     */
    public static function obfuscate (String $plainBase64): string
    {
        $bitsForChar64 = [];
        $charsForBits64 = [];
        for ($b = 0; $b < 65; $b ++) {
            $character = substr(base64charsPlus, $b, 1);
            $charsForBits64[$b] = $character;
            $bitsForChar64[$character] = $b;
        }
        $xor64 = "";
        // the key must not contain a padding character ('=')
        $kLen = strlen(self::obfuscator);
        $pLen = strlen($plainBase64);
        $k = 0;
        for ($p = 0; $p < $pLen; $p ++) {
            $ki = $bitsForChar64[substr(self::obfuscator, $k, 1)];
            $pi = $bitsForChar64[substr($plainBase64, $p, 1)];
            // do not xor the padding part.
            if ($pi == 64)
                $xor64 .= "=";
            else
                $xor64 .= $charsForBits64[$pi ^ $ki];
            $k ++;
            if ($k == $kLen)
                $k = 0;
        }
        return $xor64;
    }

    /**
     * Encode the timestamp + validity and the user Mail to create a user login token. It will have the user
     * mail in the middle, braced by two changing parts, the timestamp and a padding. The result will be a
     * base64 encoded String in which three characters are replace in order to be URL-compatible: "=" by "_",
     * "/" by "-", "+" by "*".
     */
    public static function createLoginToken (String $user_mail, int $validity, String $deep_link): string
    {
        $message = (microtime(true) + $validity * 24 * 3600) . "::" . $user_mail . "::" . $deep_link . "::" .
            substr(str_shuffle(base64charsPlus), 0, 16);
        Runner::getInstance()->logger->log(LoggerSeverity::INFO, "DilboIds::createLoginToken",
            "created: " . $message);
        return str_replace("=", "_",
            str_replace("/", "-",
                str_replace("+", "*", self::obfuscate(base64_encode($message)))));
    }

    /**
     * Decode the user login token and validate it. returns an array [valid until, user mail, deep link] or false, if
     * the token is no longer valid.
     */
    public static function decodeLoginToken (String $token): array|bool
    {
        $plainText = explode("::",
            base64_decode(
                self::obfuscate(str_replace("_", "=",
                    str_replace("-", "/", str_replace("*", "+", $token))))));
        if (intval($plainText[0]) >= microtime(true))
            return $plainText;
        else
            return false;
    }

    /**
     * file name to which tokens are written
     */
    private String $tokenFile;

    /**
     * Validity period of a token in seconds
     */
    public int $tokenValidityPeriod = 1200;

    /**
     * Monitoring period of all tokens used to check, whether a user has too many tokens created.
     */
    private int $tokenMonitorPeriod = 86400;

    // tokens are monitored a full day
    /**
     * Maximum count of tokens a user can get per monitoring period.
     */
    private int $tokensAllowedInMonitorPeriod = 10;

    public function __construct ($tokenFile) { $this->tokenFile = $tokenFile; }

    /**
     * Get a new token for the user. Returns "---" if the maximum number of tokens per user and day is
     * exceeded.
     */
    public function getNewToken (int $userId): string
    {
        $usersTokenCount = $this->cleanseTokens($userId);
        if ($usersTokenCount >= $this->tokensAllowedInMonitorPeriod)
            return "---";
        $token = substr(strtoupper(Ids::generateUid(6)), 0, 6);
        if ($userId >= 0) {
            $nowSeconds = microtime(true);
            $contents = $token . ";" . $nowSeconds . ";" . $userId;
            file_put_contents($this->tokenFile, $contents, FILE_APPEND);
        }
        return $token;
    }

    /**
     * Remove all overdue tokens. Returns the count of tokens of the user with the $userId. Set $userId == 0 to count
     * all tokens.
     */
    private function cleanseTokens (int $userId): int
    {
        // read session file, split lines and check one by one
        $tokenFileIn = file_get_contents($this->tokenFile);
        $tokenFileLines = explode("\n", $tokenFileIn);
        $tokenFileOut = "";
        $nowSeconds = microtime(true);
        $usersTokensCount = 0;
        foreach ($tokenFileLines as $line) {
            $tokenParts = explode(";", $line);
            if (count($tokenParts) >= 3) {
                $period = $nowSeconds - intval($tokenParts[1]);
                if ($period < $this->tokenMonitorPeriod) {
                    // keep token, if it is within the monitoring period.
                    $tokenFileOut .= $line . "\n";
                    if (($userId == intval($tokenParts[1])) || ($userId == 0))
                        $usersTokensCount++;
                }
            }
        }
        // write cleansed file.
        file_put_contents($this->tokenFile, $tokenFileOut);
        return $usersTokensCount;
    }

    /**
     * Reads a token file and return the userId of the matching token's user. Return -1 on no match.
     */
    public function getUserId ($token): Int
    {
        $this->cleanseTokens(0);
        // read tokens file, split lines and check one by one
        $tokenFileIn = file_get_contents($this->tokenFile);
        $tokenFileLines = explode("\n", $tokenFileIn);
        // Identify user for this session first.
        $nowSeconds = microtime(true);
        foreach ($tokenFileLines as $line) {
            $tokenParts = explode(";", $line);
            if (count($tokenParts) >= 3) {
                $period = $nowSeconds - intval($tokenParts[1]);
                if (($period < $this->tokenValidityPeriod) && (strcasecmp($token, $tokenParts[0]) == 0))
                    return intval($tokenParts[2]);
            }
        }
        return -1;
    }
}
