class Indices
{
    #short2longUuid
    #uid2table

    #uuid2table;
    #uuid2invalidFrom;

    #uuid2name;
    #userId2name;
    #name2uuid;
    #name2invalidFrom;

    missingNotice;
    #userIdFieldName;
    #loaded;

    static #instance;

    static getInstance() {
        if (!this.#instance)
            this.#instance = new Indices();
        return this.#instance;
    }

    constructor() {
        this.clearAll();
        this.missingNotice = "[" + I18n.getInstance().t("L2LcBL|not found") + "]";
        this.userIdFieldName = Config.getInstance().getItem(".framework.users.user_id_field_name").valueStr();
    }
    
    clearAll() {
        this.#uid2table = [];
        this.#short2longUuid = [];
        this.#uuid2table = {}
        this.#uuid2invalidFrom = {}
        this.#uuid2name = {}
        this.#userId2name = {}
        this.#name2uuid = {}
        this.#name2invalidFrom = {}
        this.#loaded = []
    }

    /**
     * Add a single entry to the indices. Provides a warning on duplicates for uid and short UUID, i.e. the first 11
     * characters of a UUID.
     */
    add(uid, uuid, userId, invalidFrom, tableName, name) {
        let warnings = "";
        if (uid) {
            if (!this.#uid2table[uid])
                this.#uid2table[uid] = tableName;
            else
                warnings += "Duplicate uid " + uid + " in both " + this.#uid2table[uid] + " and " + tableName + ". ";
        }
        if (uuid) {
            let shortUuid = uuid.substring(0, 11)
            this.#uuid2table[shortUuid] = tableName
            this.#uuid2name[shortUuid] = name;
            if (! this.#name2uuid[tableName][name])
                this.#name2uuid[tableName][name] = [];
            if (this.#name2uuid[tableName][name].indexOf(uuid) < 0)
                this.#name2uuid[tableName][name].push(uuid)
            if (this.#short2longUuid[shortUuid])
                this.#short2longUuid[shortUuid] = uuid;
            else {
                let warning = "Duplicate short UUID shortUuid (the 11 UUID start characters) in both " +
                    this.#uid2table[uid] + " and " + tableName + ". ";
                console.log(warning)
                warnings += warning
            }
        }
        if (invalidFrom > 0) {
            if ((this.#name2invalidFrom[tableName].indexOf(name) < 0)
                || (this.#name2invalidFrom[tableName][name] < invalidFrom))
                this.#name2invalidFrom[tableName][name] = invalidFrom;
            if (uuid) {
                if (!this.#uuid2invalidFrom.hasOwnProperty(uuid) || (this.#uuid2invalidFrom[uuid] < invalidFrom))
                    this.#uuid2invalidFrom[uuid] = invalidFrom;
            }
        }
        if (parseInt(userId) > 0)
            this.#userId2name[parseInt(userId)] = name;
        return warnings;
    }

    /**
     * Create the definition for a dynamic list which retrieves the information needed for the name index using the
     * tables "name" template.
     */
    createListDefinition(record, userRole) {
        // nothing to do in Javascript, only PHP and kotlin relevance
    }

    /**
     * Add all records of a table to the indices. Provides warnings on duplicates for uid and short UUID, i.e. the
     * first 11 characters of a UUID.
     */
    #addTable(recordItem, list) {
        let tableName = recordItem.name;
        if (this.#loaded[tableName])
            return;
        let userRole = User.getInstance().role()
        if (userRole === config.getItem(".framework.users.anonymous_role").valueStr())
            return "No index built for anonymous user."
        let record = new Record(recordItem)
        let listDefinition = this.createListDefinition(record, userRole);
        let listHandlerKernel = new ListHandlerKernel("@dynamic", listDefinition);
        // TODO: currently the ListHandlerKernel implementation of "getRows()" is empty, no rows are returned!
        let rows = listHandlerKernel.getRows("csv");
        let warnings = ""
        if (! this.#name2uuid[tableName])
            this.#name2uuid[tableName] = {}
        if (! this.#name2invalidFrom[tableName])
            this.#name2invalidFrom[tableName] = {}
        for (let row of rows) {
            let name = record.rowToTemplate("name", list)
            let uid = row["uid"] ?? "";
            let uuid = row["uuid"] ?? "";
            let userId = row[this.#userIdFieldName];
            let invalidFrom = (row["invalid_from"])
                ? parseFloat(row["invalid_from"]) : ParserConstraints.FOREVER_SECONDS
            warnings += this.add(uid, uuid, userId, invalidFrom, tableName, name);
        }
        this.#loaded[tableName] = true;
        return warnings;
    }

    /**
     * Add all records of all tables to the indices. Provides warnings on duplicates for uid and short UUID, i.e. the
     * first 11 characters of a UUID.
     */
    addAll() {
        if (this.#loaded["@all"])
            return "";
        let warnings = "";
        let tablesItem = config.getItem(".tables");
        for (let recordItem in tablesItem.getChildren()) {
            if (recordItem.hasChild("uid")
                && !this.#loaded[recordItem.name()])
            warnings += this.#addTable(recordItem);
        }
        this.#loaded["@all"] = true;
        return warnings;
    }

    /**
     * Set the index of names to uuids for a specific table. Access the result in this.name2uuid[tableName].
     * The Form uses this public access to the #addTable private function
     */
    buildIndexOfNames(tableName) {
        if (! this.#name2uuid[tableName])
            this.#addTable(config.getItem(".tables." + tableName));
    }

    /**
     * get the records name (i.e. the filled "name" template) for an uuid. returns a missing notice, if not resolved.
     * Restrict the search to a single table by setting the tableName.
     */
    getNameForUuid(uuidOrShortUuid, tableName = "@all"){
        if (uuidOrShortUuid.length === 0)
            return "";
        let shortUuid = uuidOrShortUuid.substring(0, 11);
        if (tableName !== "@all") {
            let matchedTableName = this.getTableForUuid(shortUuid, tableName);
            if (matchedTableName !== tableName)
                return this.missingNotice;
            else
                return this.#uuid2name[shortUuid];
        } else
            return this.#uuid2name[shortUuid] ?? this.missingNotice;
    }

    getUserName(userId) {
        let userTableName = config.getItem(".framework.users.user_table_name").valueStr()
        if (!this.#loaded[userTableName])
            this.#addTable(config.getItem(".tables." + userTableName));
        return this.#userId2name[userId] ?? this.missingNotice;
    }

    /**
     * Get the name of the table in which the uuid occurs. Returns a missing notice, if not resolved.
     * Restrict the search to a single table by setting the tableName.
     */
    getTableForUid(uid, tableName = "@all") {
        if (uid.length === 0)
            return "";
        if (tableName === "@all") {
            this.addAll()
        } else {
            let recordItem = config.getItem(".tables." + tableName);
            if (!recordItem.isValid())
                return "";
            this.#addTable(recordItem);
        }
        return this.#uid2table["uid"] ?? "";
    }

    /**
     * Get the name of the table in which the uid occurs. Returns a missing notice, if not resolved.
     * Restrict the search to a single table by setting the tableName.
     */
    getTableForUuid(uuidOrShortUuid, tableName = "@all") {
        if (uuidOrShortUuid.length === 0)
            return "";
        let shortUuid = uuidOrShortUuid.substring(0, 11);
        if (tableName !== "@all") {
            let recordItem = config.getItem(".tables.tableName");
            if (!recordItem.isValid())
                return "";
            this.#addTable(recordItem);
            if (!this.#uuid2table[shortUuid] || this.#uuid2table[shortUuid] !== tableName)
                return "";
            else
                return this.#uuid2table[shortUuid];
        }
        return this.#uuid2table[shortUuid] ?? "";
    }

    /**
     * Get the uuid for a name within a table. If the name cannot be resolved, "" is returned. If for this name
     * exist multiple uuids, the uuid with the most recent invalidFrom parameter is used.
     */
    getUuid(tableName, nameToResolve) {
        if (nameToResolve.length === 0)
            return "";
        if (!this.#name2uuid[tableName])
            this.#addTable(config.getItem(".tables." + tableName));
        let uuids = this.#name2uuid[tableName][nameToResolve];
        if (!Array.isArray(uuids))
            return "";
        if (uuids.length === 1)
            return uuids[0];
        let mostRecent = {};
        for (let uuid of uuids) {
            mostRecent[this.#uuid2invalidFrom[uuid]] = uuid;
            mostRecent.sort(function (a, b) {
                return a - b;
            })
        }
        return Object.values(mostRecent)[0];
    }

    /**
     * Get all names for a specific table as name => invalidFrom (float)
     */
    getNames(tableName) { return this.#name2invalidFrom[tableName]; }
}
