/**
 * tools-for-your-hobby
 * https://www.tfyh.org
 * Copyright  2023-2025  Martin Glade
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

const i18nReplacementRegex = [ /%1/g, /%2/g, /%3/g, /%4/g, /%6/g, /%7/g, /%8/g, /%9/g ]
/**
 * A generic translation helper
 */
class I18n {
		
	// localization
	static #instance = new I18n();
	#i18nURI = "../i18n/#.lrf";
	#map = [];
	loaded = false;

	static getInstance() { return this.#instance }

	isValidI18nReference(toCheck)
	{
		if ((toCheck.length < 7) || (toCheck.substring(6, 7) !== "|"))
			return false
		else if (!this.loaded)
			return false
		else
			return (typeof this.#map[toCheck.substring(0, 6)] !== 'undefined');
	}

	parseLrf(lrf) {
		let lines = lrf.split(/\n/g);
		let token = false;
		let text = "";
		for (let line of lines) {
			if (line.indexOf("|") !== 6)
				text += "\n" + line;
			else {
				if (token)
					this.#map[token] = text;
				token = line.substring(0, 6);  
				text = line.substring(7);
			}
		}
	}
	
	/**
	 * load the i18n data. This is asynchronous. Use te callback function to continue.
	 */
	loadResource (localeToUse, callback) {
		this.#i18nURI = this.#i18nURI.replace("#", localeToUse)
		// prepare the post request.
		let getRequest = new XMLHttpRequest();
		getRequest.timeout = 10000; // milliseconds
		let that = this
		// provide the callback for a response received
		getRequest.onload = function() {
			that.parseLrf(getRequest.response);
			that.loaded = true;
			callback();
		};
		// provide the callback for any error
		getRequest.onerror = function() {
			alert("Fatal error loading application texts for internationalization. Texts will be empty");
			that.loaded = true;
			callback();
		};
		// provide the callback for timeout
		getRequest.ontimeout = function() {
			alert("Fatal error loading application texts for internationalization. Texts will be empty");
			that.loaded = true;
			callback();
		};
		// send the post request
		getRequest.open('GET', this.#i18nURI);
		getRequest.send();
	}

	/**
	 * Call this function to get the proper translation of your texts. Up to 5 non-translatable arguments can be used
	 * within the text.
	 */
	t(...args) {
		if (args.length === 0)
			return "";
		let i18nResource = args[0]
		if ((i18nResource.length < 7) || (i18nResource.substring(6, 7) !== "|"))
			return i18nResource;
		let token = i18nResource.substring(0, 6);
		let text = "";
		if (!this.loaded)
			text = i18nResource.substring(7);
		else
			text = this.#map[token] ?? i18nResource.substring(7);
		if (typeof text !== 'string')
			// if the key is a valid identifier of an own property of the Array
			// prototype it can not be resolved, but will return the Array prototype
			// function or whatever.
			return "[" + i18nResource + "]";
		// if the mapping results in an empty string (missing translation) us the resource reference
		if (text.length === 0)
			return i18nResource.substring(7);
		for (let i = 1; i < args.length; i++)
			text = text.replace(i18nReplacementRegex[i], args[i]);
		return text;
	}
}

